


/* =========================================================================
   Trainset Framework - Jon Wolfers - 2015
   A bit of fun for the 26th International Rexx Symposium

   Classes
   =======

   Line         -- A trainline
   Sector       -- Part of a line that may contain a station
                   and host an up and/or dn train
   Station      -- A train station
   Train        -- A train
   Passenger    -- A passenger

   Supporting Classes
   ==================
   Visualise    -- creates a bitmap from the current state of a line

   TrainLineGui -- Provides a context to run the train line in


  Requires imageMagick from
   Train icon by http://www.vectorgraphit.com/
   ========================================================================= */




/* ========================================================================= */
::requires 'oodwin32.cls'    -- for visualisation
::requires 'winsystm.cls'    -- for registry access (locate imageMagick)
/* ========================================================================= */




/* =====----================================================================ */
::class Line                     public               -- trainset line class
/* =====----================================================================ */
::attribute name             -- name of the line
::attribute colour           -- colour associated with line
::attribute tick
::attribute demand           -- 0 - 100 how busy line is
-- collections --
::attribute sectors          -- array of sectors making up line
::attribute stations         -- directory of stations on the line by name
::attribute stationArr       -- arr of sectorIds of open stations
::attribute trains           -- directory of trains   on the line by name
-- statistics
::attribute passengersTotal         -- total passengers on     system
::attribute passengersOnTrain       -- total passengers on     trains
::attribute CompletedJourneySectors -- total passenger-sectors  completed
::attribute CompletedJourneys       -- total passenger-journeys completed
::attribute CompletedJourneyTicks   -- total passenger-ticks    completed

/* ------------------------------------------------------------------------- */
::method init
/* ------------------------------------------------------------------------- */
expose sectors stations colour trains stationArr
use arg self~name, numSectors

   colour = 'blue'   -- default track colour

   sectors = .array~new(numSectors)              -- create the sectors
   do i = 1 to numSectors
      sectors[i] = .Sector~new(i,self)
   end /* DO */

   sectors[1]~isDnTerminus?          = .true     -- ends of the line
   sectors[numSectors]~isUpTerminus? = .true

   stations   = .directory~new                   -- directory for stations
   trains     = .directory~new                   -- directory for trains
   stationArr = .array~new                       -- sector numbers of open stations

   self~demand  = 50                             -- average day for pax numbers

   self~reset                                    -- set stats to zero

/* ------------------------------------------------------------------------- */
::method addStation
/* ------------------------------------------------------------------------- */
expose stations sectors stationArr
use arg name, sectorId

   if sectors[sectorId]~station \= .nil
   then return ErrorDialog('Cannot add station ['name'] at sector' sectorId 'as station ['sectors[sector]~station~name'] is already there')

   sector  = sectors[sectorId]
   station = .station~new(name,sector,self)
   sector~station = station
   stations[name]            = station
   station~capacity          = .station~capacity -- default capacity
   station~minPax            = .station~minPax   -- default range
   if sector~isTerminus?                         -- termini have fewer pax
   then station~maxPax       = (.station~maxPax + .station~minPax) % 2
   else station~maxPax       = .station~maxPax   -- default range
   stationArr~append(sectorId)

return .true
/* ------------------------------------------------------------------------- */
::method addTrain
/* ------------------------------------------------------------------------- */
expose sectors
use arg name, sectornum, direction

   sectors[sectornum]~addTrain(name,direction) -- pass this task on to sector

/* ------------------------------------------------------------------------- */
::method ToggleBiDirectional
/* ------------------------------------------------------------------------- */
expose sectors
use arg start, end

   do i = min(start,end) to max(start,end)
      sector = sectors[i]
      if   sector~isA(.Sector)
      then sector~bidirectional? = \sector~bidirectional?
   end /* DO */

/* ------------------------------------------------------------------------- */
::method run
/* ------------------------------------------------------------------------- */
expose sectors tick

   if arg(1,'e')
   then tick = arg(1)
   else if tick~datatype(w)
        then tick += 1
        else tick = 1

-- run each line against the flow of the trains
   do i = 1 to sectors~last
      sectors[i]~run(tick,'dn')
   end /* DO */
   do i = sectors~last to 1 by -1
      sectors[i]~run(tick,'up')
   end /* DO */

/* ------------------------------------------------------------------------- */
::method reset
/* ------------------------------------------------------------------------- */
/* clear all passengers and statistics                                       */
expose stations trains passengersTotal passengersOnTrain CompletedJourneys ,
       CompletedJourneySectors CompletedJourneyTicks Tick

   if tick~datatype('n') then say self~string

   do station over stations~allItems
      station~upPax~empty
      station~dnPax~Empty
      station~exitingPax~empty
      station~waitingPax~empty
   end /* DO */

   do train over trains~allItems
      train~passengers~empty
      train~embarked       = 0
      train~disembarked    = 0
   end /* DO */

   passengersTotal         = 0
   passengersOnTrain       = 0
   CompletedJourneys       = 0
   CompletedJourneySectors = 0
   CompletedJourneyTicks   = 0

   Tick                    = 0

/* ------------------------------------------------------------------------- */
::method string
/* ------------------------------------------------------------------------- */
expose passengersTotal passengersOnTrain CompletedJourneySectors              -
       CompletedJourneys CompletedJourneyTicks name demand tick
   summary = .array~of(name,
                      ,'Tick' Tick,
                      ,'Demand:' demand'%',
                      ,'Passengers:' passengersTotal,
                      ,'% on trains:' format((passengersOnTrain/max(PassengersTotal,1)) * 100,,2)||'%',
                      ,'Completed Journeys:' CompletedJourneys,
                      ,'Ticks per Sector' format(completedJourneyTicks / max(completedJourneySectors,1),,2),
                      ,'')

   trArray = .array~of('=== Trains ===',
                      ,'Name         Station    Sector up?  passengers capacity')
   do trainName over self~trains
      train = self~trains[trainName]
      if train~sector~station~isA(.station)
      then location = train~sector~station~name~left(12) train~sector~id~right(4)
      else location = train~sector~id~right(17)
      upBit = 'X'~copies(train~up?)
      trArray~append(trainName~left(12) location~left(17) upBit~center(4) train~passengers~items~right(10) train~capacity~right(8))
   end /* DO */

   stArr   = .array~of('=== Stations ===',
                      ,'Name         pax:  up       dn  exiting  waiting')
   do stationName over self~stations
      station = self~stations[stationName]
      stArr~append(stationName~left(12) station~upPax~items~right(8) station~dnPax~items~right(8) station~exitingPax~items~right(8) station~waitingPax~items~right(8))
   end /* DO */

return summary~union(trArray~union(stArr))~makeString
/* ========================================================================= */





/* =====------============================================================== */
::class Sector
/* =====------============================================================== */
::attribute id                -- id is index of line~sectors
::attribute station           -- station if present
::attribute upTrain           -- upTrain if present
::attribute dnTrain           -- dnTrain if present
::attribute line              -- line this sector is part of
::attribute biDirectional?    -- single track?
::attribute lastTrainThrough  -- [up,dn] needed to manage bidirectional conflict
::attribute isUpTerminus?     -- end of up line?
::attribute isDnTerminus?     -- end of dn line?
::attribute isTerminus? get
expose isUpTerminus? isDnTerminus?
return isUpTerminus? | isDnTerminus?

/* ------------------------------------------------------------------------- */
::method init
/* ------------------------------------------------------------------------- */
expose station upTrain dnTrain biDirectional? isDnTerminus? isUpTerminus?
use arg self~id, self~line

   station        = .nil
   upTrain        = .nil
   dnTrain        = .nil
   biDirectional? = .false
   isDnTerminus?  = .false
   isUpTerminus?  = .false

/* ------------------------------------------------------------------------- */
::method addTrain
/* ------------------------------------------------------------------------- */
expose line
use arg name, direction = 'up'

   up? = 'up'~caselessAbbrev(direction)
   train = .train~new(line,name,up?,self)

   if up?
   then self~upTrain = train
   else self~dnTrain = train

   line~trains[name] = train

/* ------------------------------------------------------------------------- */
::method nextUp
/* ------------------------------------------------------------------------- */
expose line id
return line~sectors[id + 1]

/* ------------------------------------------------------------------------- */
::method nextDn
/* ------------------------------------------------------------------------- */
expose line id
if id = 1
then return .nil
else return line~sectors[id - 1]

/* ------------------------------------------------------------------------- */
::method run
/* ------------------------------------------------------------------------- */
expose upTrain dnTrain Station
use arg tick, dir

   if station~isA(.station) then station~run(tick)
   if dir = 'up'
   then if upTrain~isA(.train)
        then upTrain~run(tick)
        else nop
   else if dnTrain~isA(.train)
        then dnTrain~run(tick)
        else nop

/* ========================================================================= */





/* =====-------============================================================= */
::Class station
/* =====-------============================================================= */
/* ===================== Class Methods ===================================== */
::attribute Capacity CLASS -- default capacity
::attribute MinPax   CLASS -- default min Pax per tick
::attribute MaxPax   CLASS -- default max pax per tick
/* ------------------------------------------------------------------------- */
::method init        CLASS
/* ------------------------------------------------------------------------- */
   self~Capacity = 400
   self~minPax   =  10
   self~maxPax   =  40

/* ====================== Instance methods ================================= */
::attribute name           -- station name
::attribute sector         -- sector where station is situated
::attribute line           -- line station is on
::attribute capacity       -- platform Capacity
::attribute minPax         -- least no of pax starting journey per tick
::attribute maxPax         -- most  no of pax starting journey per tick (demand may double this)
::attribute demand         -- How busy is station (0-100)
::attribute closing?       -- is the station closing?
-- collections             --
::attribute upPax          -- queue of passengers on up platform
::attribute dnPax          -- queue of passengers on dn platform
::attribute exitingPax     -- queue of passengers just detrained
::attribute waitingPax     -- set of pax waiting to get on a platform (because it is full)
/* ------------------------------------------------------------------------- */
::method init
/* ------------------------------------------------------------------------- */
use arg self~name, self~sector, self~line

   self~upPax       = .queue~new
   self~dnPax       = .queue~new
   self~exitingPax  = .queue~new
   self~waitingPax  = .set~new

   self~closing?    = .false
   self~capacity    = .station~capacity  -- default platform capacity
   self~demand      = 50

/* ------------------------------------------------------------------------- */
::method run
/* ------------------------------------------------------------------------- */
expose line closing? minPax maxPax sector upPax dnPax waitingPax capacity
use arg tick

-- deal with pax waiting to get onto (previously) overcrowded platform
   do pax over waitingPax
      if pax~up?
      then if upPax~items < capacity
           then upPax~push(waitingPax~remove(pax))
           else nop
      else if dnPax~items < capacity
           then dnPax~push(waitingPax~remove(pax))
           else nop
   end /* DO */

   if closing? then return

-- accept new pax off the street
   paxNum = (random(minPax,maxPax) * (line~demand / 50) * (self~demand / 50)) % 1
   do paxNum
      pax = .passenger~new(line,sector~id)
      if pax~up?
      then if   upPax~items < capacity
           then upPax~push(pax)
           else waitingPax~put(pax)
      else if   dnPax~items < capacity
           then dnPax~push(pax)
           else waitingPax~put(pax)
   end /* DO */

/* ========================================================================= */








/* =====-----=============================================================== */
::class train
/* =====-----=============================================================== */
::attribute capacity CLASS
/* ------------------------------------------------------------------------- */
::method init        CLASS
/* ------------------------------------------------------------------------- */
   self~capacity = 700  -- default train capacity
/* -------------------- instance methods ----------------------------------- */
::attribute Name get       -- name of the train
::attribute up?            -- currently on up track?
::attribute Sector         -- sector train is on
::attribute line           -- line train is on
::attribute disabled       -- 0 = no, +ve = ticks remaining, -ve = indefinite
::attribute blocked?       -- set by move .true means train blocked
::attribute capacity       -- no of people that fit on train
::attribute outOfService?  -- take out of service at next station
::attribute stationTask    -- ['none','embark','depart']
::attribute embarked       -- no pax embarked last move
::attribute disembarked    -- no pax disembarked last move
-- collection --           --
::attribute passengers     -- relation of passengers by destination
/* ------------------------------------------------------------------------- */
::method init
/* ------------------------------------------------------------------------- */
expose name
use arg self~line, name, self~up?, self~sector

   self~disabled      = 0
   self~blocked?      = .false
   self~outOfService? = .false
   self~passengers    = .relation~new
   self~stationTask   = 'none'
   self~capacity      = .train~capacity  -- default capacity
   self~embarked      = 0
   self~disembarked   = 0

/* ------------------------------------------------------------------------- */
::method run                                     -- each tick runs this
/* ------------------------------------------------------------------------- */
expose up? disabled sector stationTask capacity passengers outOfService? line ,
        embarked disembarked
use arg tick
--say 'running train' self~name

   thisSector  = sector         -- for later comparison
   embarked    = 0
   disembarked = 0

   if disabled > 0             -- disabled for a particular no of ticks
   then do
      disabled -= 1
      return
   end /* DO */

   if disabled < 0 then return -- disabled indefinitely

-- we are not disabled
   if sector~station~isA(.station)
   then do
      station = sector~station
      select
         when stationTask = 'none' -- just arrived - disembark pax
            then do
               disembarkers = passengers~allAt(sector~id)
               disembarked = disembarkers~items
               do pax over disembarkers
                  station~exitingPax~push(pax)
                  passengers~removeItem(pax)
                  pax~endTick = tick
                  pax~debrief(line)
               end /* DO */
               if outOfService? = .true
               then do
                  disembarked += passengers~items
                  do pax over passengers
                     if up?
                     then station~upPax~push(passengers~remove(pax))
                     else station~dnPax~push(passengers~remove(pax))
                  end
                  -- train will be deleted on departure
               end /* DO */

               if sector~isUpTerminus? = .true,  -- no embarkation at terminus
               |  sector~isDnTerminus? = .true,
               |  outOfService?        = .true
               then stationTask = 'depart'
               else stationTask = 'embark'
            end /* DO */
         when stationTask = 'embark'
            then do
               if up?
               then paxQueue = station~upPax
               else paxQueue = station~dnPax

               do while passengers~items < capacity
                  if paxQueue~items = 0 then leave
                  pax = paxQueue~pull
                  passengers~put(pax,pax~destination)
                  line~passengersOnTrain += 1
                  embarked += 1
               end /* DO */

               stationTask = 'depart'
            end /* DO */
         when stationTask = 'depart'
            then do
               if outOfService? & passengers~items = 0
               then do  -- remove train
                  if up?
                  then sector~upTrain = .nil
                  else sector~dnTrain = .nil
                  line~trains~remove(self~name)
                  sector = .nil
                  return
               end /* DO */

               self~move

               if sector \= thisSector -- we have managed to leave
               then do
                  stationTask = 'none'
               end /* DO */
            end /* DO */
         otherwise
            raise syntax 88.900 array('StationTask must be one of [none,embark,depart] found:' stationtask)
      end /* select */
   end /* DO */
   else self~move

/* ------------------------------------------------------------------------- */
::method move
/* ------------------------------------------------------------------------- */
expose up? sector blocked? disabled
--say 'moving train' self~name 'in sector' sector~id

   blocked? = .false                -- default position
   if disabled  \= 0 then return    -- cant move because we are disabled

   if up?
   then self~moveUp
   else self~moveDn

/* ------------------------------------------------------------------------- */
::method moveUp
/* ------------------------------------------------------------------------- */
expose sector

   if sector~isUpTerminus?          then RETURN self~reverse
   if sector~nextUp~upTrain \= .nil then RETURN self~blocked('Train Ahead')
   if \sector~biDirectional?, sector~nextUp~bidirectional?
   then do
      -- we are entering a single track space
      -- if there is an oncoming train on or waiting to enter, we stay
      target = sector
      do while target~nextUp~biDirectional? = .true
         if target~nextUp~dnTrain \= .nil
         then if target~nextUp~biDirectional? ,
              |  sector~nextUp~lastTrainThrough = 'up' -- play nice if deadlock
              then RETURN self~blocked('Train on single track section')
         target = target~nextUp
      end /* DO */
   end /* DO */

   -- OK to move
   sector~lastTrainThrough = 'up'
   target         = sector~nextUp
   target~upTrain = self
   sector~upTrain = .nil
   sector         = target

/* ------------------------------------------------------------------------- */
::method moveDn
/* ------------------------------------------------------------------------- */
expose sector

   if sector~isDnTerminus?          then RETURN self~reverse
   if sector~nextDn~dnTrain \= .nil then RETURN self~blocked('Train Ahead')
   if \sector~biDirectional?, sector~nextDn~bidirectional?
   then do
      -- we are entering a single track space
      -- if there is an oncoming train on or waiting to enter, we stay
      target = sector
      do while target~nextDn~biDirectional? = .true
         if target~nextDn~upTrain \= .nil
         then if target~nextDn~biDirectional? ,
              |  sector~nextDn~lastTrainThrough = 'dn' -- play nice if deadlock
              then RETURN self~blocked('Train on single track section')
         target = target~nextDn
      end /* DO */
   end /* DO */

   -- OK to move
   sector~lastTrainThrough = 'dn'
   target         = sector~nextDn
   target~dnTrain = self
   sector~dnTrain = .nil
   sector         = target

/* ------------------------------------------------------------------------- */
::method blocked
/* ------------------------------------------------------------------------- */
expose blocked?
use arg reason

 blocked? = .true
 --   report a blockage

return blocked?
/* ------------------------------------------------------------------------- */
::method Reverse    -- move from up line to dn line or vica versa
/* ------------------------------------------------------------------------- */
expose up? sector stationTask

   if up?
   then do
      if sector~dnTrain = .nil
      then do
         sector~dnTrain = self
         sector~uptrain = .nil
         up? = .false
      end /* DO */
      else self~blocked('Cannot reverse - line in use')
   end /* DO */
   else do
      if sector~upTrain = .nil
      then do
         sector~upTrain = self
         sector~dntrain = .nil
         up? = .true
         self~disabled = 1 -- stop train skipping first block on up track
      end /* DO */
      else self~blocked('Cannot reverse - line in use')
   end /* DO */

   if sector~station~isA(.station) then stationTask = 'embark'

return 0
/* ========================================================================= */





/* =====---------=========================================================== */
::class passenger
/* =====---------=========================================================== */
::attribute origin
::attribute destination
::attribute starttick
::attribute endTick
::attribute up?
/* ========================================================================= */
::method init
/* ------------------------------------------------------------------------- */
expose origin destination startTick up?
use arg line, origin

   stationArr = line~stationArr
   select
      when stationArr~items = 0 then Return
      when stationArr~items = 1 then destination = origin
      otherwise
         do until destination \= origin
            destination = stationArr[random(1,stationArr~last)]
         end /* DO */
   end /* select */

   up? = (destination > origin)
   startTick = line~tick
   line~passengersTotal += 1               -- record this passengers inception

/* ------------------------------------------------------------------------- */
::method deBrief
/* ------------------------------------------------------------------------- */
expose origin destination startTick endTick
use arg line

   line~passengersTotal         -= 1
   line~passengersOnTrain       -= 1
   line~completedJourneySectors += abs(origin  - destination)
   line~completedJourneys       += 1
   line~CompletedJourneyTicks   += (endtick - startTick)

/* ========================================================================= */







/* =====---------=========================================================== */
::class visualise               public
/* =====---------=========================================================== */
::attribute imBinPath CLASS  -- location of the imageMagick install
::attribute trainImg  CLASS
::attribute tmpDir    CLASS
/* ------------------------------------------------------------------------- */
::method init         CLASS
/* ------------------------------------------------------------------------- */
expose imBinPath trainImg tmpDir

   /* Get the imageMagick path from the registry */
   myReg = .WindowsRegistry~new
   if myReg~InitCode \= 0
   then Raise syntax 48.900 Array('Error opening the registry. Could not fetch imageMagick path')

   node = myReg~open(myReg~local_machine,'SOFTWARE\ImageMagick\Current','READ')
   if node = 0
   then Raise syntax 98.900 Array('This script requires ImageMagick from http://www.imagemagick.org/')

   a. = myReg~getValue(node,'BinPath')
   imBinPath = a.data||'\'

   drop myReg

   /* specify path to image trains are created from */
   parse source . . sourceFile
   /* Train icon by http://www.vectorgraphit.com/ */
   trainImg = filespec('L',sourceFile)||'trainset.png'

   /* temporary folder to house images */
   tmpDir = value('TMP',,'environment')||'\'


/* ------------------------------------------------------------------------- */
::attribute script -- array for ImageMagickScript
::attribute line
::attribute trainImages
::attribute Image
::attribute stroke set
expose stroke script
use arg newStroke
   if \script~isA(.Array) then raise syntax 93.900 array('Cannot set stroke VALUE when no Script exists')
   if newStroke \= stroke
   then do
      stroke = newStroke
      script~append('stroke' stroke)
   end /* DO */

::attribute fill   set
expose fill   script
use arg newFill
   if \script~isA(.Array) then raise syntax 93.900 array('Cannot set fill VALUE when no Script exists')
   if newFill \= fill
   then do
      fill = newfill
      script~append('fill' fill)
   end /* DO */

-- attributes required for gui to calculate mouse click sector and bar
::attribute bars         public
/* ------------------------------------------------------------------------- */
::method init
/* ------------------------------------------------------------------------- */
expose line image_width image_height tmpDir imDir trainImages
use arg line, image_width = 1200, image_height = 600

   imDir = .visualise~imBinPath

   if \line~isA(.line)            then raise syntax 88.900 array ('Train line expected, found' line~class)
   if \image_width~datatype('w')  then raise syntax 26.900 array ('Image Width must be a positive whole number')
   if \image_height~datatype('w') then raise syntax 26.900 array ('Image Height must be a positive whole number')

   tmpDir = value('TMP',,'environment')||'\'

   trainImages = .directory~new

/* ------------------------------------------------------------------------- */
::method show_line
/* ------------------------------------------------------------------------- */
expose line script image_width image_height sector_width bars bar_height      -
       imDir composites

   script     = .array~new
   composites = .array~new
   bars       = .array~of('title','station','upplatform','uptrain','line','dntrain','dnplatform')
   sector_width = image_width  % (line~sectors~items + 1)
   bar_height   = image_height % (bars~items + 1)

   xM           = image_width  % 2
   yT           = self~bar_y('title') + 1
-- put name of line in title
   self~stroke = line~colour
   self~fill   = line~colour
   script~append('font-size 25')
   script~append(im_Text( 5,yT    ,line~name))
   script~append(im_Text(xM,yT,'Tick' line~tick '- Demand' line~demand||'%'))
   script~append(im_text( 5,yT + 26, 'Passengers in system:' right(line~PassengersTotal,8)))
   script~append(im_text(xM,yT + 26, 'Percentage on Trains:' format((line~passengersOnTrain/max(line~PassengersTotal,1)) * 100,,2)))
   script~append(im_text( 5,yT + 52, 'Completed Journeys  :' right(line~CompletedJourneys,8)))
   script~append(im_text(xM,yT + 52, 'Avg Ticks per sector:' format(line~completedJourneyTicks / max(line~completedJourneySectors,1),,2)))
   script~append('font-size 15')

-- mark the ends of the track
   end_Mark_top = self~bar_y('line') + (bar_height % 4)
   end_Mark_bot = end_Mark_top  + (bar_height % 2)
   self~stroke = line~colour
   script~append(im_Line(self~sector_x(1),end_Mark_top,,end_Mark_bot))
   script~append(im_Line(self~sector_x(line~sectors~items + 1),end_Mark_top,,end_Mark_bot))

   do sector over line~sectors
      self~script_Sector(script,sector)
   end /* DO */


   /* prepare an imageMagick drawing command file */
   imCmdFile  = .visualise~tmpDir||'drawcmds.mvg'
   self~image = imCmdFile~changeStr('.mvg','.bmp')
   imCmdStream  = .stream~new(imCmdFile)
   imCmdStream~open('write replace')
   imCmdStream~arrayOut(script)
   imCmdStream~close

   cmd = '"'||imDir||'convert" -size' image_width||'x'||image_height,
                     ' xc:darkgray'       ,
                     '-fill red'          ,
                     '-font Arial'        ,
                     '-pointsize 15'      ,
                     '-draw @'||imcmdFile ,
                     composites~makeString('l',' '),
                     'BMP3:'self~image '> nul'       -- ooDialog requires BMP3 variant of BMP
-- say cmd
   cmd

/* ------------------------------------------------------------------------- */
::method script_Sector
/* ------------------------------------------------------------------------- */
expose bar_height sector_width track_divide
use arg script, sector

   sectorId = sector~id     -- do we need this?
   line    = sector~line   -- do we need this?

   -- mark the sector track
   sector_left  = self~sector_X(sectorId) + 1
   sector_right = sector_left + sector_width - 2
   sector_mid_y = self~bar_y('line','Centre')
   track_divide = 5              -- pixels above/below centre for dual track

   self~stroke = line~colour
   self~fill   = line~colour
   if sector~bidirectional?
   then do
      script~append(im_Line(sector_left,sector_mid_y,sector_right))

      if sectorId > line~sectors~first, line~sectors[sectorId - 1]~biDirectional? = .false
      then self~script_fletch_left( script, sector)

      if sectorId < line~sectors~last , line~sectors[sectorId + 1]~biDirectional? = .false
      then self~script_fletch_right(script, sector)

   end /* DO */
   else do
      script~append(im_Line(sector_left,sector_mid_y - track_divide,sector_right))
      script~append(im_Line(sector_left,sector_mid_y + track_divide,sector_right))
   end /* DO */


   if sector~station~isA(.station) then self~script_station(script, sector)
   if sector~upTrain~isA(.train)   then self~script_Train(script, sector, sector~upTrain)
   if sector~dnTrain~isA(.train)   then self~script_Train(script, sector, sector~dnTrain)

/* ------------------------------------------------------------------------- */
::method script_station
/* ------------------------------------------------------------------------- */
expose image_height
use arg script, sector

   station = sector~station
   sectorId = sector~id

   self~script_box(script, sector, 'station', 'lightgrey')
   self~script_box(script, sector, 'upplatform', 'lightgrey', -17)
   self~script_box(script, sector, 'dnplatform', 'lightgrey',  15)

   self~stroke = sector~line~colour
   script~append(im_Text(self~sector_x(sectorId) + 5, self~bar_y('station') + 12, 'Station'))
   script~append(im_Text(self~sector_x(sectorId) + 5, self~bar_y('station') + 30, station~name))
   script~append('circle' self~sector_x(SectorId,'Centre')','self~bar_y('line','centre') self~sector_x(SectorId,'Centre')','self~bar_y('line','centre')+10)

   if station~waitingPax~items > 0
   then do
      self~stroke = 'red'
      script~append(im_Text(self~sector_x(sectorId) + 5, self~bar_y('station') + 50, station~waitingPax~items))
   end /* DO */

   if station~upPax~items = station~capacity
   then self~stroke = 'red'
   else self~stroke = sector~line~colour

   if station~upPax~items > 0
   then script~append(im_Text(self~sector_x(sectorId) + 5, self~bar_y('upplatform') + 14, station~upPax~items 'Pax.'))

   if station~dnPax~items = station~capacity
   then self~stroke = 'red'
   else self~stroke = sector~line~colour

   if station~dnPax~items > 0
   then script~append(im_Text(self~sector_x(sectorId) + 5, self~bar_y('dnplatform','bottom') - 12, station~dnPax~items 'Pax.'))

/* ------------------------------------------------------------------------- */
::method script_train
/* ------------------------------------------------------------------------- */
expose bar_height sector_width imDir composites trainImages
use arg script, sector, train

   if \trainImages~HasIndex(train~name)    -- first time this is run for this train
   then do
      dlg = .dlgSplash~new(train~name)

         trainImages[train~name] = .directory~new
         do mode over .array~of('Running','Stopped','Clamped')

         -- up
            target   = SysTempFileName(.visualise~tmpDir||'train????.png')
            cmd = imDir||'convert' .visualise~trainImg~changeStr('set.',mode||'.')                                   -
                                   '-flop '                                   -
                                   '-resize' sector_width||'x'||bar_height'!' -
                                   '-background orange '                      -
                                   'label:"'train~name'" '                    -
                                   '-gravity Center '                         -
                                   '-append' target
            cmd
            trainImages[train~name]['up_'||mode] = target

         -- dn
            target   = SysTempFileName(.visualise~tmpDir||'train????.png')
            cmd = imDir||'convert' .visualise~trainImg~changeStr('set.',mode||'.')                                      -
                                   '-resize' sector_width||'x'||bar_height'!'    -
                                   '-background orange '                         -
                                   'label:"'train~name'"'                        -
                                   '-gravity Center '                            -
                                   '-append' target
            cmd
            trainImages[train~name]['dn_'||mode] = target
      end /* DO */
   end /* DO */

   if train~up?
   then track = 'up'
   else track = 'dn'

   sectorId = sector~id
   if track = 'up'
   then geo = '+'||self~sector_x(sectorId)||'+'||self~bar_y('uptrain')
   else geo = '+'||self~sector_x(sectorId)||'+'||(self~bar_y('dntrain')-15)

   select
      when train~disabled \= 0                then image = trainImages[train~name][track||'_Clamped']
      when train~sector~station~isA(.station) then image = trainImages[train~name][track||'_Stopped']
      otherwise                                    image = trainImages[train~name][track||'_Running']
   end /* select */

   composites~append(image '-geometry' geo '-composite')

   if train~passengers~items > 0
   then do

      if train~passengers~items = train~capacity
      then self~stroke = 'red'
      else self~stroke = sector~line~colour

      if train~up?
      then script~append(im_Text(self~sector_x(sectorId) + 5, self~bar_y('uptrain') - 5, train~passengers~items 'Pax.'))
      else script~append(im_Text(self~sector_x(sectorId) + 5, self~bar_y('dntrain','bottom') + 12, train~passengers~items 'Pax.'))
   end /* Do */

   if train~embarked > 0
   then if train~up?
        then script~append(im_Text(self~sector_x(sectorId) + 5, self~bar_y('upplatform','centre') - 5, center('v' train~embarked 'v',7)))
        else script~append(im_Text(self~sector_x(sectorId) + 5, self~bar_y('dnplatform','centre') - 5, center('^' train~embarked '^',7)))

   if train~disembarked > 0
   then if train~up?
        then script~append(im_Text(self~sector_x(sectorId) + 5, self~bar_y('upplatform','centre') - 5, center('^' train~disembarked '^',7)))
        else script~append(im_Text(self~sector_x(sectorId) + 5, self~bar_y('dnplatform','centre') - 5, center('v' train~disembarked 'v',7)))

   if dlg~isA(.dlgSplash) then dlg~cancel

/* ------------------------------------------------------------------------- */
::method script_box
/* ------------------------------------------------------------------------- */
expose bar_height
use arg script, sector, bar, colour = '', blank = 0

   if blank > 0
   then offTop = blank
   else offTop = 0

   if blank < 0
   then offBot = blank
   else offBot = 0

   sectorId = sector~id
   if colour \= ''
   then do
      self~stroke = colour
      self~fill   = colour
   end /* DO */

   box_y = self~bar_y(bar)
   script~append('rectangle' self~sector_x(sectorId)||','||(box_y + offTop) (self~sector_x(sectorId + 1) + 1)||','||(box_y + bar_height - 2 + offBot))


/* ------------------------------------------------------------------------- */
::method script_fletch_left        private
/* ------------------------------------------------------------------------- */
expose track_divide
use arg script, sector

    self~stroke = sector~line~colour
    self~fill   = sector~line~colour
    sectorId  = sector~id
    fletch_x = self~sector_x(sectorId) - 2
    fletch_y = self~bar_y('line','Centre')

    script~append(im_Line(fletch_x, fletch_y - track_divide, fletch_x + 4, fletch_y))
    script~append(im_Line(fletch_x, fletch_y + track_divide, fletch_x + 4, fletch_y))

/* ------------------------------------------------------------------------- */
::method script_fletch_right       private
/* ------------------------------------------------------------------------- */
expose track_divide
use arg script, sector

    self~stroke = sector~line~colour
    self~fill   = sector~line~colour
    sectorId  = sector~id + 1
    fletch_x = self~sector_x(sectorId) - 2
    fletch_y = self~bar_y('line','Centre')

    script~append(im_Line(fletch_x, fletch_y, fletch_x + 4, fletch_y - track_divide))
    script~append(im_Line(fletch_x, fletch_y, fletch_x + 4, fletch_y + track_divide))

/* ------------------------------------------------------------------------- */
::method sector_x            private
/* ------------------------------------------------------------------------- */
/* passed a sector number returns the x value for the cell left edge         */
expose sector_width
if arg(2,'e'),'CENTRE'~caselessAbbrev(arg(2))
then offset = 0
else offset = 0.5
return trunc(((arg(1) - offset) * sector_width) + 0.5)

/* ------------------------------------------------------------------------- */
::method bar_y               private
/* ------------------------------------------------------------------------- */
/* passed a bar number returns the y value for the cell top edge             */
expose bar_height bars

   barId = bars~index(arg(1))

   if arg(2,'e'), 'Bottom'~caselessAbbrev(arg(2)) then barId += 1

   if barId = .nil
   then raise syntax 88.900 array ('Bar Name expected, found ['arg(1)'] not from ['bars~makeString('l',',')']')

   y = trunc(((barId - 0.5) * bar_height) + 0.5)

if arg(2,'e'), 'Centre'~caselessAbbrev(arg(2))
then return y + (bar_height % 2)  -- centre of bar

return y

/* ------------------------------------------------------------------------- */
::method cleanUp
/* ------------------------------------------------------------------------- */
/* If files have been created for the visualisation of train - delete them   */
expose trainImages

   if trainImages~hasMethod('MAKEARRAY')
   then do train over trainImages
      if trainImages[train]~hasMethod('MAKEARRAY')
      then do image over trainImages[train]
         call sysFileDelete trainImages[train][image]
      end /* DO */
   end /* DO */

   call sysfiledelete .visualise~tmpDir||'drawcmds.mvg'
   call sysfiledelete .visualise~tmpDir||'drawcmds.bmp'

/* ========================================================================= */



/* ========================================================================= */
::routine im_line        public
/* ========================================================================= */
use arg x1, y1, x2, y2

   if arg(3,'o') then x2 = x1  -- allow for vertical   lines
   if arg(4,'o') then y2 = y1  -- allow for horizontal lines

return 'line' x1','y1 x2','y2
/* ========================================================================= */

/* ========================================================================= */
::routine im_text       public
/* ========================================================================= */
use arg x1, y1, text
return 'text' x1','y1 '"'text'"'
/* ========================================================================= */




/* ========================================================================= */
::class trainLineGUI public subclass userDialog inherit advancedControls -
                                                        messageExtensions
/* ========================================================================= */
::attribute tickNo  unguarded
::attribute Stop    unguarded
::attribute ood4plus
::method init
/* ------------------------------------------------------------------------- */
expose line visualiser tickNo stop ood4plus
use arg line

   visualiser = .visualise~new(line,1320,660)
   visualiser~show_line

   self~init:super
   width=890 ; height=435

   self~createCenter(width,height,'TrainSet by Rexx')

   tickNo = 0

   stop = .false

-- ooDialog 4.2 introduced an incompatibility
   ood4plus = .false
   parse value .dlgutil~version with major'.'minor'.'
   if major > 4 | (major = 4 & minor > 1)
   then ood4plus = .true


/* ------------------------------------------------------------------------- */
::method defineDialog
/* ------------------------------------------------------------------------- */
expose BMPFile

   sx = self~sizeX ; sy = self~sizeY
   BMPFile = .visualise~tmpDir||'drawcmds.bmp'


   self~addBitMapButton( 100,           10,    10, sx-20, sy-41,'Bitmap','onImageClick',BMPFile)

   self~addButton(       101,           10, sy-25,    50,    20, 'Run'  ,'RunIt'   ,'DEFAULT')
   self~addText(                        65, sy-19,    15,    10, 'for')
   self~addEntryLine(    102,'ticks',   75, sy-22,    20,    15, 'NUMBER CENTER')
   self~addText(                       100, sy-19,    15,    10, 'ticks.')

   self~addButton(       103,       sx-120, sy-25,    50,    20, 'Reset','Reset')
   self~addButton(       104,          125, sy-25,    50,    20, 'Pause', 'PauseOrResume','Disabled')

   self~addButton(         2,        sX-60, sy-25,    50,    20, 'Leave','Cancel')
   self~addText(                       200, sy-19,    50,    10, 'Quiet','Right')
   self~addSliderControl(105,'Demand' ,255, sy-25,   120,    20)
   self~addText(                       380, sy-19,    50,    10, 'Busy')

/* ------------------------------------------------------------------------- */
::method initDialog
/* ------------------------------------------------------------------------- */
expose stopBut line demandSlide

   self~ticks = 10
   stopBut    = self~getButtonControl(104)

   self~ConnectSliderNotify(105,"Drag","onDemandChange")
   demandSlide  = self~getSliderControl(105)

   demandSlide~initRange(0,100)
   demandSlide~setTickFrequency(10)
   self~demand = line~demand

/* ------------------------------------------------------------------------- */
::method RunIt unguarded
/* ------------------------------------------------------------------------- */
expose tickNo line BMPFile visualiser stop stopBut sessionTick ood4plus
reply                           -- return to dialog to watch for stop button

   self~getData
   stopBut~title = 'Pause'
   stopBut~enable

   if arg(1,'e'),arg(1) = 'RESUME'
   then start = sessionTick
   else start = 1

   do sessionTick = Start to self~ticks
      if stop = .true
      then do
         if sessionTick < self~ticks then stopBut~title = 'Resume'
         RETURN                                                            -->|
      end /* DO */
      tickNo += 1
      line~run(tickNo)
      visualiser~show_line

   -- ooDialog 4.2 introduced an incompatibility
      if ood4plus
      then self~changeBitmapButton(100,BMPFile,,,,'DRAW')
      else self~changeBitmapButton(100,BMPFile)

   end /* DO */
   stop = .false

   stopBut~title = 'Pause'
   stopBut~disable

/* ------------------------------------------------------------------------- */
::method PauseOrResume unguarded
/* ------------------------------------------------------------------------- */
expose stop stopBut sessionTick

   stopBut~Style="DEFPUSHBUTTON"
   if stopBut~Title = 'Pause'
   then do
      stop = .true
      stopBut~Title = 'Resume'
   end /* DO */
   else do
      stopBut~Title = 'Pause'
      stop = .false
      self~runIt('RESUME')
   end /* DO */

/* ------------------------------------------------------------------------- */
::method onImageClick
/* ------------------------------------------------------------------------- */
/* handle clicks on the image                                                */
expose visualiser line BMPFile ood4plus

-- Map the click onto a bar and sector ----------------------------------------
-- get button rectangle
   parse value self~getButtonRect(100) with bx1 by1 bx2 by2
   bCx = bx2 - bx1
   bCy = by2 - by1
-- get cursor position
   parse value self~CursorPos with curx cury
-- adjust so cursor origin is 0,0 for button
   curX -= bx1
   curY -= by1
-- sector_width on button is not the same as sector_width on BMP so recalculate it
   sector_width = bCx % (line~sectors~last + 1)
-- calculate sector
   sectorNum    = (curX + (sector_width % 2)) % sector_width
-- bar height on button is not the same as bar_height on BMP so recalculate it
   bar_height   = bCy % (visualiser~bars~last + 1)
   barNum       = (curY + (bar_height % 2)) % bar_height

   if barNum    < 1,
   |  barNum    > visualiser~bars~last
   then RETURN            -- click outside area

   bar          = visualiser~bars[barNum]
-------------------------------------------------------------------------------

   if sectorNum < 1,
   |  sectorNum > line~sectors~last,
   |  bar = .nil
   then RETURN            -- click outside area

-- handle the click -----------------------------------------------------------

   select
--    when bar = 'title' then
      when bar = 'station'    ,
         | bar = 'upplatform' ,
         | bar = 'dnplatform'
         then changed? = self~editStation(sectorNum)
      when bar = 'uptrain'    ,
         | bar = 'dntrain'
         then changed? = self~editTrain(sectorNum,bar)
      when bar = 'line'       then changed? = self~editSector(sectorNum)
      otherwise    changed? = .false
   end /* select */

   if changed? = .true
   then do
      visualiser~show_line                          -- update image
      if ood4plus
      then self~changeBitmapButton(100,BMPFile,,,,'DRAW')
      else self~changeBitmapButton(100,BMPFile)
   end /* DO */

/* ------------------------------------------------------------------------- */
::method editTrain
/* ------------------------------------------------------------------------- */
expose line visualiser
use arg sectorNum, bar

   dlg = .dlgEditTrain~new(line,sectorNum,bar~caselessAbbrev('up'), visualiser)
   dlg~execute('showtop')

return (dlg~InitCode = 1)

/* ------------------------------------------------------------------------- */
::method editSector
/* ------------------------------------------------------------------------- */
expose line
use arg sectorNum

   dlg = .dlgEditSector~new(line~sectors[sectorNum])
   dlg~execute('showtop')

return (dlg~InitCode = 1)

/* ------------------------------------------------------------------------- */
::method editStation
/* ------------------------------------------------------------------------- */
expose line
use arg sectorNum

   dlg = .dlgEditStation~new(line~sectors[sectorNum])
   dlg~execute('showtop')

return (dlg~InitCode = 1)

/* ------------------------------------------------------------------------- */
::method onDemandChange
/* ------------------------------------------------------------------------- */
expose line demandSlide
self~getData

   line~demand = self~demand

/* ------------------------------------------------------------------------- */
::method reset
/* ------------------------------------------------------------------------- */
expose line visualiser BMPFile ood4plus

   line~reset
   self~tickNo = 0
   visualiser~show_line                          -- update image

   if ood4plus
   then self~changeBitmapButton(100,BMPFile,,,,'DRAW')
   else self~changeBitmapButton(100,BMPFile)      -- show updated image

/* ------------------------------------------------------------------------- */
::method cancel
/* ------------------------------------------------------------------------- */
expose visualiser stop

   stop = .true       -- do not process any more ticks
   visualiser~cleanup -- delete temporary train images etc.
   self~cancel:super  -- tear down dialog

return self~finished
/* ========================================================================= */




/* ========================================================================= */
::class dlgEditTrain subclass userdialog inherit AdvancedControls
/* ========================================================================= */
::attribute changed?
/* ------------------------------------------------------------------------- */
::method Init
/* ------------------------------------------------------------------------- */
expose width height line sectorNum up? train visualiser
use arg line, sectorNum, up?, visualiser

   self~Init:super()
   width = 270 ; height = 130

   if up?
   then train = line~sectors[sectorNum]~upTrain
   else train = line~sectors[sectorNum]~dnTrain

   if train~isA(.train)
   then title = 'Edit a train'
   else title = 'Add a train'

   rc=self~CreateCenter(width,height,title)
   self~InitCode=(rc=0)

/* ------------------------------------------------------------------------- */
::method DefineDialog
/* ------------------------------------------------------------------------- */
expose width height

   labX  = 75
   labCx = width - labX
   self~AddButton(     1,        width-60,height-20,      50,15,'OK','Ok','DEFAULT')
   self~AddButton(     2,       width-120,height-20,      50,15,'Cancel','Cancel')
-- self~AddButton(     9,        width-60,height-60,      50,15,'Help','Help')
   self~AddEntryLine(110,'Name',       10,       10,      50,10)
   self~AddText(                     labX,       10,   labCx,10,'Name of train')
   self~addEntryLine(111,'Capacity',   10,       25,      20,15,'NUMBER CENTER')
   self~AddText(                     labX,       25,   labCx,10,'Capacity')
   self~AddEntryLine(112,'SectorNum',  10,       40,      25,10,'NUMBER CENTER READONLY')
   self~AddText(                     labX,       40,   labCx,10,'Sector')
   self~AddEntryLine(113,'Disabled',   10,       55,      20,15,'CENTER')
   self~AddText(                     labX,       55,   labCx,10,'No of ticks disabled for. -1 is indefinite')
   self~AddCheckBox( 114,'up?',        10,       85,     100,10,'up track?','Disabled')
   self~AddCheckBox( 115,'OOS?',       10,      100,     100,10,'Take out of service?')

/* ------------------------------------------------------------------------- */
::method initdialog                                                            /*{3.02}*/
/* ------------------------------------------------------------------------- */
expose line sectorNum up? train

   if train~isA(.train)
   then do
      self~name      = train~name
      self~sectorNum = sectorNum
      self~disabled  = train~disabled
      self~up?       = up?
      OOS?           = train~outOfService?
      self~capacity  = train~capacity
   end /* DO */
   else do           -- we are adding a new one
      self~name      = ''
      self~sectorNum = sectorNum
      self~up?       = up?
      self~disabled  = 0
      self~OOS?      = .false
      self~capacity = .train~capacity -- default train capacity
   end /* DO */

/* ------------------------------------------------------------------------- */
::method Validate
/* ------------------------------------------------------------------------- */
expose line

   self~GetData

   select
      when self~name~length > 12
         then   valid? = errordialog("Train name can only be 12 characters")
      when \self~capacity~datatype('w') | self~capacity < 0
         then   valid? = errorDialog("Capacity must be a whole number")
      when \self~sectorNum~datatype('w')
         then   valid? = errorDialog("Sector must be a whole number between 1 and" line~sectors~last)
      when \line~sectors[self~sectorNum]~isA(.Sector)
         then   valid? = errorDialog("Invalid Sector ["self~sectorNum"]")
      otherwise valid? = .true
   end /* select */

return valid?

/* ------------------------------------------------------------------------- */
::method ok
/* ------------------------------------------------------------------------- */
expose line changed? visualiser

   self~OK:super

   changed? = .false
   if self~finished = 1
   then do
      train = line~trains[self~name]
      if \train~isA(.train)
      then do
         line~addTrain(self~name,self~sectorNum,.array~of('dn','up')[self~up? + 1])
         train = line~trains[self~name]
         train~disabled = self~disabled
         train~capacity = self~capacity
      end /* DO */
      else do

         if self~name \= train~name
         then do
            changed? = .true
            train~name = self~name
         end /* DO */

         if self~sectorNum \= train~sector~id,
         |  self~up? \= train~up?
         then do
            train~up?    = self~up?
            if train~up?
            then do
               line~sectors[self~sectorNum]~upTrain = train
               train~sector~upTrain = .nil
            end /* DO */
            else do
               line~sectors[self~sectorNum]~dnTrain = train
               train~sector~dnTrain = .nil
            end /* DO */
            train~sector = line~sectors[self~sectorNum]
         end /* DO */

         train~disabled      = self~disabled
         train~outOfService? = self~OOS?
         train~capacity      = self~capacity
      end /* DO */
   end /* DO */

   if self~finished & changed?
   then do
      do file over visualiser~trainImages[train~Name]
         call sysFileDelete file
      end /* DO */
      visualiser~trainImages[train~Name] = .nil -- force redraw of train images
   end /* DO */

return self~finished

/* ========================================================================= */



/* ========================================================================= */
::class dlgEditSector subclass userdialog inherit AdvancedControls
/* ========================================================================= */
::attribute changed?
/* ------------------------------------------------------------------------- */
::method Init
/* ------------------------------------------------------------------------- */
expose width height sector changed?
use arg sector

   self~Init:super()
   width = 270 ; height = 45

   rc=self~CreateCenter(width,height,'Edit track sector' sector~id)

   self~InitCode=(rc=0)
   changed? = .false

/* ------------------------------------------------------------------------- */
::method DefineDialog
/* ------------------------------------------------------------------------- */
expose width height

   self~AddButton(     1,        width-60,height-20,      50,15,'OK','Ok','DEFAULT')
   self~AddButton(     2,       width-120,height-20,      50,15,'Cancel','Cancel')

   self~AddCheckBox( 100,'bi?',        10,       10,     100,10,'BiDirectional?')

/* ------------------------------------------------------------------------- */
::method initdialog                                                            /*{3.02}*/
/* ------------------------------------------------------------------------- */
expose sector

   if sector~isA(.Sector) then self~bi? = sector~bidirectional?

/* ------------------------------------------------------------------------- */
::method ok
/* ------------------------------------------------------------------------- */
expose sector changed?

   self~OK:super

   self~getData
   if self~finished = 1 & sector~biDirectional? \= self~bi?
   then do
      sector~biDirectional? = self~bi?
      changed? = .true
   end /* DO */

return self~finished

/* ========================================================================= */



/* ========================================================================= */
::class dlgEditStation subclass userdialog inherit AdvancedControls -
                                                   MessageExtensions
/* ========================================================================= */
::attribute changed?
/* ------------------------------------------------------------------------- */
::method Init
/* ------------------------------------------------------------------------- */
expose width height sector changed?
use arg sector

   self~Init:super()
   width = 270 ; height = 140

   if sector~station~isA(.station)
   then tag = 'Edit Station' sector~station~name
   else tag = 'Add Station'

   rc=self~CreateCenter(width,height, tag)

   self~InitCode=(rc=0)
   changed? = .false

/* ------------------------------------------------------------------------- */
::method DefineDialog
/* ------------------------------------------------------------------------- */
expose width height

   self~AddButton(     1,        width-60,height-20,      50,15,'OK','Ok','DEFAULT')
   self~AddButton(     2,       width-120,height-20,      50,15,'Cancel','Cancel')

   labX  = 120
   labCx = width - labX
   cntX  = labX - 55
   self~AddEntryLine(    110,'Name',      cntX,       10,      50, 10)
   self~addText(                          labX,       10,   labCX, 10, "Name")
   self~AddEntryLine(    111,'SectorNum', cntX,       25,      50, 10,'Readonly')
   self~addText(                          labX,       25,   labCX, 10, "Sector")
   self~AddEntryLine(    112,'Capacity',  cntX,       40,      50, 10)
   self~addText(                          labX,       40,   labCX, 10, "Platform Capacity")
   self~addEntryLine(    113,'MinPax',    cntX,       55,      20, 10,'Number right')
   self~addText(                      cntX +20,       55,      10, 10,'-','center')
   self~addEntryLine(    114,'MaxPax',cntX +30,       55,      20, 10,'Number left')
   self~addText(                          labX,       55,   labCx, 10,'Range of passengers')
   self~addText(                          labX,       65,   labCx, 10,'entering per tick.')
   self~addSliderControl(116,'demand',      10,       80,     105, 10)
   self~addText(                          labX,       80,   labCx, 10, 'Demand Quiet <-> Busy')
   self~AddCheckBox(     115,'closing?',labX-17,     105,     100, 10, '   Close Station?')


/* ------------------------------------------------------------------------- */
::method initdialog                                                            /*{3.02}*/
/* ------------------------------------------------------------------------- */
expose sector

   self~sectorNum = sector~id

   if sector~station~isA(.station)
   then do
      station        = sector~station

      self~name      = station~name
      self~Capacity  = station~capacity
      self~minPax    = station~minPax
      self~maxPax    = station~maxPax
      self~closing?  = station~closing?
      self~demand    = station~demand
   end /* DO */
   else do           -- we are adding a new one
      self~name      = 'New Station'
      self~capacity  = .station~capacity -- apply the default
      self~minPax    = .station~minPax
      self~maxPax    = .station~maxPax
      self~closing?  = .false
      self~demand    = 50
   end /* DO */



/* ------------------------------------------------------------------------- */
::method Validate
/* ------------------------------------------------------------------------- */
expose sector

   self~GetData

   select
      when self~name~length > 12
         then   valid? = errordialog("Station name can only be 12 characters")
      when \self~sectorNum~datatype('w')
         then   valid? = errorDialog("Sector must be a whole number between 1 and" line~sectors~last)
      when \self~capacity~datatype('w')
         then   valid? = errorDialog("Capacity must be a +ve whole number, not ["self~capacity"]")
      when self~capacity < 0
         then   valid? = errorDialog("Capacity must be a +ve whole number, not ["self~capacity"]")
      when \self~minPax~datatype('w')
         then   valid? = errorDialog("Passenger range lower bound  must be a +ve whole number, not ["self~capacity"]")
      when self~minPax < 0
         then   valid? = errorDialog("Passenger range lower bound  must be a +ve whole number, not ["self~capacity"]")
      when \self~maxPax~datatype('w')
         then   valid? = errorDialog("Passenger range lower bound  must be a +ve whole number, not ["self~capacity"]")
      when self~maxPax < 0
         then   valid? = errorDialog("Passenger range lower bound  must be a +ve whole number, not ["self~capacity"]")
      when self~minPax > self~maxPax
         then      valid? = errorDialog("Passenger range lower bound  must be less than upper bound.")
      otherwise valid? = .true
   end /* select */

return valid?

/* ------------------------------------------------------------------------- */
::method ok
/* ------------------------------------------------------------------------- */
expose sector changed?

   self~OK:super

   if self~finished = 1
   then do
      if \sector~station~isA(.station)
      then changed? = sector~line~addStation(self~name,self~sectorNum)
      station = sector~station

      if (self~name     \= station~name    ),
      |  (self~capacity \= station~capacity),
      |  (self~minPax   \= station~minPax  ),
      |  (self~maxPax   \= station~maxPax  ),
      |  (self~demand   \= station~demand  ),
      |  (self~closing? \= station~closing?)
      then do
         changed? = .true
         station~name     = self~name
         station~capacity = self~capacity
         station~minPax   = self~minPax
         station~maxPax   = self~maxPax
         station~demand   = self~demand
         station~closing? = self~closing?
      end /* DO */
      if station~closing?
      then sector~line~stationArr = sector~line~stationArr~removeItem(sector~id)~makeArray
      else if   sector~line~stationArr~index(sector~Id) = 0
           then sector~line~stationArr~append(sector~Id)
   end /* DO */

return self~finished

/* ========================================================================= */




/* ========================================================================= */
::class dlgSplash subclass userdialog
/* ========================================================================= */
::method Init
/* ------------------------------------------------------------------------- */
expose trainname width
use arg trainName

   self~Init:super()
   width = 200 ; height = 30

   rc=self~CreateCenter(width,height, "Trainset - please wait")

   self~InitCode=(rc=0)
   self~executeAsync                       -- run automatically till cancelled

/* ------------------------------------------------------------------------- */
::method DefineDialog
/* ------------------------------------------------------------------------- */
expose trainname width

   self~addText(10,10, width - 20,15,'Preparing train' trainname 'for service','center')

/* ------------------------------------------------------------------------- */
::method cancel
/* ------------------------------------------------------------------------- */
   ret = self~cancel:super
   self~endAsyncExecution

return ret
/* ========================================================================= */
